const STORAGE_KEY = "realruntime_gemini_demo_config_v1";
const SECRET_STORAGE_KEY = "realruntime_gemini_demo_secrets_v1";

const refs = {
  realruntimeBase: document.getElementById("realruntimeBase"),
  requiredScope: document.getElementById("requiredScope"),
  geminiBase: document.getElementById("geminiBase"),
  model: document.getElementById("model"),
  realruntimeKey: document.getElementById("realruntimeKey"),
  geminiKey: document.getElementById("geminiKey"),
  emotionalGoal: document.getElementById("emotionalGoal"),
  intensity: document.getElementById("intensity"),
  intensityValue: document.getElementById("intensityValue"),
  checkSession: document.getElementById("checkSession"),
  saveConfig: document.getElementById("saveConfig"),
  sendMessage: document.getElementById("sendMessage"),
  messageInput: document.getElementById("messageInput"),
  sessionStatus: document.getElementById("sessionStatus"),
  chatStream: document.getElementById("chatStream"),
};

function removeStatusClasses() {
  refs.sessionStatus.classList.remove("ok", "err");
}

function setStatus(message, kind = "neutral") {
  refs.sessionStatus.textContent = message;
  removeStatusClasses();
  if (kind === "ok") refs.sessionStatus.classList.add("ok");
  if (kind === "err") refs.sessionStatus.classList.add("err");
}

function clamp(value, min, max) {
  return Math.min(Math.max(value, min), max);
}

function parseJsonSafe(raw, fallback = {}) {
  try {
    return JSON.parse(raw);
  } catch (_err) {
    return fallback;
  }
}

function formatIntensity(value) {
  return `${Number(value).toFixed(2)} / 1.00`;
}

function refreshIntensityLabel() {
  refs.intensityValue.textContent = formatIntensity(refs.intensity.value);
}

function readConfigFromUi() {
  return {
    realruntime_api_base: refs.realruntimeBase.value.trim(),
    required_scope: refs.requiredScope.value.trim(),
    gemini_api_base: refs.geminiBase.value.trim(),
    model: refs.model.value.trim(),
    emotional_goal: refs.emotionalGoal.value,
    intensity: clamp(Number(refs.intensity.value || 0), 0, 1),
  };
}

function readSecretsFromUi() {
  return {
    realruntime_api_key: refs.realruntimeKey.value.trim(),
    gemini_api_key: refs.geminiKey.value.trim(),
  };
}

function writeConfigToUi(config) {
  if (!config) return;
  if (config.realruntime_api_base) refs.realruntimeBase.value = config.realruntime_api_base;
  if (config.required_scope) refs.requiredScope.value = config.required_scope;
  if (config.gemini_api_base) refs.geminiBase.value = config.gemini_api_base;
  if (config.model) refs.model.value = config.model;
  if (config.emotional_goal) refs.emotionalGoal.value = config.emotional_goal;
  if (config.intensity !== undefined && config.intensity !== null) {
    refs.intensity.value = String(clamp(Number(config.intensity), 0, 1));
  }
  refreshIntensityLabel();
}

function writeSecretsToUi(secrets) {
  if (!secrets) return;
  if (secrets.realruntime_api_key) refs.realruntimeKey.value = secrets.realruntime_api_key;
  if (secrets.gemini_api_key) refs.geminiKey.value = secrets.gemini_api_key;
}

function persistState() {
  const config = readConfigFromUi();
  const secrets = readSecretsFromUi();
  window.localStorage.setItem(STORAGE_KEY, JSON.stringify(config));
  window.sessionStorage.setItem(SECRET_STORAGE_KEY, JSON.stringify(secrets));
}

function restoreState() {
  const storedConfig = parseJsonSafe(window.localStorage.getItem(STORAGE_KEY), {});
  const storedSecrets = parseJsonSafe(window.sessionStorage.getItem(SECRET_STORAGE_KEY), {});
  writeConfigToUi(storedConfig);
  writeSecretsToUi(storedSecrets);
}

function appendMessage(role, text) {
  const message = document.createElement("article");
  message.className = `msg ${role}`;

  const roleTag = document.createElement("span");
  roleTag.className = "role";
  roleTag.textContent = role;

  const body = document.createElement("div");
  body.textContent = text;

  message.append(roleTag, body);
  refs.chatStream.appendChild(message);
  refs.chatStream.scrollTop = refs.chatStream.scrollHeight;
}

async function fetchJson(path, options = {}) {
  const res = await fetch(path, options);
  const payload = await res.json().catch(() => ({}));
  if (!res.ok) {
    const detail =
      typeof payload.detail === "string" && payload.detail.trim()
        ? payload.detail
        : `${res.status} ${res.statusText}`;
    throw new Error(detail);
  }
  return payload;
}

async function hydrateDefaults() {
  try {
    const config = await fetchJson("/v1/config");
    writeConfigToUi({
      realruntime_api_base: refs.realruntimeBase.value || config.realruntime_api_base,
      required_scope: refs.requiredScope.value || config.required_scope,
      gemini_api_base: refs.geminiBase.value || config.gemini_api_base,
      model: refs.model.value || config.model,
    });
  } catch (_err) {
    // Keep local values when backend defaults are unavailable.
  }
}

function setBusy(isBusy) {
  refs.checkSession.disabled = isBusy;
  refs.saveConfig.disabled = isBusy;
  refs.sendMessage.disabled = isBusy;
}

async function onCheckSession() {
  const { realruntime_api_key } = readSecretsFromUi();
  if (!realruntime_api_key) {
    setStatus("RealRuntime API key is required for session verification.", "err");
    return;
  }

  setBusy(true);
  setStatus("Checking RealRuntime key entitlement...", "neutral");

  try {
    const config = readConfigFromUi();
    const result = await fetchJson("/v1/session/check", {
      method: "POST",
      headers: { "content-type": "application/json" },
      body: JSON.stringify({
        realruntime_api_key,
        realruntime_api_base: config.realruntime_api_base,
        required_scope: config.required_scope,
      }),
    });

    const remMin = result.remaining_per_minute ?? "n/a";
    const remDay = result.remaining_per_day ?? "n/a";
    setStatus(
      `Verified: ${result.scope} | account ${result.account_id} | remaining ${remMin}/min ${remDay}/day`,
      "ok",
    );
    appendMessage(
      "system",
      `Key verified for scope "${result.scope}". Remaining quota: ${remMin}/minute, ${remDay}/day.`,
    );
    persistState();
  } catch (err) {
    setStatus(`Verification failed: ${err.message}`, "err");
    appendMessage("system", `Verification failed: ${err.message}`);
  } finally {
    setBusy(false);
  }
}

async function onSendMessage() {
  const message = refs.messageInput.value.trim();
  const config = readConfigFromUi();
  const secrets = readSecretsFromUi();

  if (!message) {
    setStatus("Message is required.", "err");
    return;
  }
  if (!secrets.realruntime_api_key) {
    setStatus("RealRuntime API key is required.", "err");
    return;
  }
  if (!secrets.gemini_api_key) {
    setStatus("Gemini API key is required.", "err");
    return;
  }

  appendMessage("user", message);
  refs.messageInput.value = "";
  setBusy(true);
  setStatus("Sending request through RealRuntime and Gemini...", "neutral");

  try {
    const result = await fetchJson("/v1/chat/respond", {
      method: "POST",
      headers: { "content-type": "application/json" },
      body: JSON.stringify({
        message,
        realruntime_api_key: secrets.realruntime_api_key,
        gemini_api_key: secrets.gemini_api_key,
        realruntime_api_base: config.realruntime_api_base,
        required_scope: config.required_scope,
        gemini_api_base: config.gemini_api_base,
        model: config.model,
        emotional_goal: config.emotional_goal,
        intensity: config.intensity,
      }),
    });

    appendMessage("assistant", result.answer);
    setStatus(
      `Response generated (${result.model}) with ${result.emotional_goal} @ ${Number(result.intensity).toFixed(2)}.`,
      "ok",
    );
    persistState();
  } catch (err) {
    appendMessage("system", `Request failed: ${err.message}`);
    setStatus(`Request failed: ${err.message}`, "err");
  } finally {
    setBusy(false);
  }
}

function onSaveConfig() {
  persistState();
  setStatus("Configuration saved in browser storage.", "ok");
}

function wireEvents() {
  refs.intensity.addEventListener("input", refreshIntensityLabel);
  refs.checkSession.addEventListener("click", onCheckSession);
  refs.sendMessage.addEventListener("click", onSendMessage);
  refs.saveConfig.addEventListener("click", onSaveConfig);
  refs.messageInput.addEventListener("keydown", (event) => {
    if ((event.metaKey || event.ctrlKey) && event.key === "Enter") {
      event.preventDefault();
      onSendMessage();
    }
  });
}

async function init() {
  wireEvents();
  restoreState();
  await hydrateDefaults();
  if (!refs.messageInput.value.trim()) {
    refs.messageInput.value = "Help me guide a frustrated customer toward a calm resolution.";
  }
  appendMessage(
    "system",
    "Demo ready. Run a key check first, then send chat messages. Shortcut: Ctrl+Enter (or Cmd+Enter).",
  );
}

init().catch((err) => {
  setStatus(`Initialization error: ${err.message}`, "err");
});
